import numpy as np
from numpy.linalg import norm
from scipy.optimize import minimize
import json
import sys

############################################################## Loading

def LoadPositions(filename):
	with open(filename, 'rt') as f:
		c=json.loads(f.read())

	positions = np.zeros((len(c), 3))
	orientations = np.zeros((len(c), 4))
	indices = np.zeros((len(c)), dtype=np.int)
	for i in range(len(c)):
		indices[i] = int(c[i]['id'])
		positions[i,:] = c[i]['position']
		orientations[i,:] = c[i]['rotation']
		if abs(np.linalg.norm(orientations[i,:]) - 1) > 0.01:
			raise Exception("Quaternion not normalized: " + str(orientations[i,:]))
	return positions, orientations, indices

# file names should contain the .json extension
def Load(referenceFilename, reconstructionFilename):
	refPos, refOrie, x = LoadPositions(referenceFilename)
	recPos, recOrie, i = LoadPositions(reconstructionFilename)
	completeness = len(i)/len(x)
	return refPos[i], refOrie[i], recPos, recOrie, completeness


############################################################## Comparing

def PointDistances(A, B):
	assert A.shape == B.shape
	return norm(A-B, axis=1)

def TrajectoryDistance(A, B):
	D = PointDistances(A,B)
	n = A.shape[0]
	return np.sqrt(1/n * np.sum(D**2))

# returns offset o such that A ~= B+o
def ComputeOffset(A, B):
	def f(p, A, B):
		return TrajectoryDistance(A-p, B)
	res = minimize(f, np.zeros((3)), args=(A, B))
	return res.x


# computes the distance between two quaternions q0 and q1
def RotationDistance(A, B):
	dots = np.array([1 - np.dot(A[i,:], B[i,:]) for i in range(A.shape[0])])
	return np.sqrt(1/A.shape[0] * np.sum(dots**2))

doc = \
'''command line arguments: [ReferenceFile] [SubmissionFile]
Example: python TrackingRot.py DataBase/Tracking/AirplanePathMoveRotate.json Users/ID404/AirplanePathMoveRotate.json'''
if __name__ == "__main__":
	if len(sys.argv) != 3:
		print("invalid call:\n", doc)
		sys.exit(2) # the 2 sometimes signals command line syntax errors

	refPos, refOrie, recPos, recOrie, completeness = Load(sys.argv[1], sys.argv[2])

	offset = ComputeOffset(refPos, recPos)
	originalDistance = TrajectoryDistance(refPos, recPos)
	minimizedDistance= TrajectoryDistance(refPos-offset, recPos)
	orientation_distance = RotationDistance(refOrie, recOrie)

	output = {
	"status": "success",
	"completeness": completeness,
	"offset": norm(offset),
	"original_distance": originalDistance,
	"path_distance": minimizedDistance,
	"orientation_distance": orientation_distance
	}

	print(json.dumps(output, indent="\t"))